clear all

ALLERP = buildERPstruct([]);
CURRENTERP = 0;

%% SUBJECT & EXPERIMENT PARAMETERS

Subject_ID = '01';                                          % subject ID (name of subject folder & eeg file)

Experimental_Folder = pwd;                                  % experiment folder (pwd = current folder)
Elist_File = [Experimental_Folder '\Elist_CDA.txt'];        % location & name of experiment eventlist file
Bins_File = [Experimental_Folder '\Bins_CDA.txt'];          % location & name of experiment bins file
Diff_File = [Experimental_Folder '\Difference_Waves.txt'];  % location & name of experiment difference-waves equations
Epoch_Time = [-200.0 1100.0];                               % time window (ms) for epoching
N_Conditions = 3;                                           % number of experimental conditions

%% IMPORTING PARAMETERS

Channel_Number = 72;        % number of channels recorded
Reference_Channel = 65:66;	% number of referance channel(s)

%% ARTIFACT REJECTION (MOVING WINDOW PICK-TO-PICK) PARAMETERS

N_Artifcat_Rejections = 2;                              % number of artifact rejection runs
                                                        % 1: cleaning only the VEOG & HEOG channels; 
                                                        % 2: cleaning other channels as well (e.g. the to-be-analyzed channels)

% 1st artifact detection:
Moving_Window_Threshold_1 = 80;                         % threshold (in mV) for the 1st artifact detection
Moving_Window_Width_1 = 200;                            % window width (ms) for the 1st artifact detection
Moving_Window_Step_Size_1 = 100;                        % window step (ms) for the 1st artifact detection
Moving_Window_Electrodes_1 = [67:69];                   % numbers of electrodes on which to perform the 1st artifact detection

% 2nd artifact detection:
Moving_Window_Threshold_2 = 100;                        % threshold for the 2nd artifact detection
Moving_Window_Width_2 = 200;                            % window width (ms) for the 2nd artifact detection
Moving_Window_Step_Size_2 = 100;                        % window step (ms) for the 2nd artifact detection
Moving_Window_Electrodes_2 = [23 25 26 27 60 62 63 64];	% numbers of electrodes on which to perform the 2nd artifact detection

%% FILTERING PARAMETERS

Highpass_Filter = 0;        % apply a highpass filter?
                            % 0 = no (only a lowpass filter for the epoched data)
                            % 1 = yes (highpass filter the continous data)

Lowpass_Cutoff = 30;        % lowpass filter cutoff in Hz
Lowpass_Type = 'butter';    % lowpass filter type
Lowpass_Order = 2;          % lowpass filter length in points

Highpass_Cutoff = 0.1;      % highpass filter cutoff in Hz
Bandpass_Type = 'butter';	% highpass filter type
Bandpass_Order = 2;         % highpass filter length in points

%% PLOTTING PARAMETERS

Plot_Types = 2;                                     % figures to create
                                                    % 1 = present only difference waves
                                                    % 2 = present ipsi & contra plots as well

Bins_to_Plot_1 = (N_Conditions*2+1):N_Conditions*3;	% bins to plot - difference waves
Bins_to_Plot_2 = 1:(N_Conditions*2);                % bins to plot - ipsi & contra

Channels_to_Plot = [1:4];                           % numbers of channels that would be presented in the figure

%% SAVING & WARNINGS PARAMETERS

Save_All = 1;       % save all sets?
                    % 0 = save only the first and last set; 1 = save all sets
Warnings = 'off';	% warnings on or off?

%% Analyze

fprintf('\nProcessing subject %s\n\n', Subject_ID);
    
Subject_Path=[Experimental_Folder '\' Subject_ID '\'];

% import/load the data file
if exist([Subject_Path Subject_ID '.bdf'], 'file') >0
    EEG = pop_biosig([Subject_Path Subject_ID '.bdf'], 'ref', Reference_Channel);
    EEG.setname = [Subject_ID '_EEG'];
    EEG = eeg_checkset(EEG);
    EEG = pop_saveset(EEG, 'filename', [EEG.setname '.set'], 'filepath', Subject_Path);
    EEG = eeg_checkset(EEG);
    
    % create eventlist
    fprintf('\n%s: Creating eventlist\n', Subject_ID);
    EEG = pop_editeventlist(EEG, 'BoundaryNumeric', {-99}, 'BoundaryString', {'boundary'},...
        'ExportEL', [Subject_Path Subject_ID '_elist.txt'], 'List', Elist_File, 'SendEL2', 'EEG&Text', 'UpdateEEG', 'on', 'Warning', Warnings);
    EEG = eeg_checkset(EEG);
    EEG.setname = [EEG.setname '_elist'];
    if (Save_All)
        EEG = pop_saveset(EEG, 'filename', [EEG.setname '.set'], 'filepath', Subject_Path);
    end        
    
    % highpass filter (only if Highpass_Filter is set to 1)
    if Highpass_Filter
        EEG  = pop_basicfilter( EEG,  1:Channel_Number , 'Cutoff', [Highpass_Cutoff Lowpass_Cutoff],...
            'Design', Bandpass_Type, 'Filter', 'bandpass', 'Order',  Bandpass_Order );
        EEG = eeg_checkset( EEG );
        EEG.setname = [EEG.setname '_filt'];
        if (Save_All)
            EEG = pop_saveset(EEG, 'filename', [EEG.setname '.set'], 'filepath', Subject_Path);
        end
    end   
    
    % sort bins
    fprintf('\n%s: Sorting bins\n', Subject_ID);
    EEG = pop_binlister(EEG, 'BDF', Bins_File, 'ExportEL', [Subject_Path Subject_ID '_bins.txt'],...
        'ImportEL', 'no', 'Saveas', 'on', 'SendEL2', 'EEG&Text',...
        'UpdateEEG', 'on', 'Warning', Warnings);
    EEG = eeg_checkset(EEG);
    EEG.setname = [EEG.setname '_bins'];
    if (Save_All)
        EEG = pop_saveset(EEG, 'filename', [EEG.setname '.set'], 'filepath', Subject_Path);
    end
        
    % bin-based epochs
    fprintf('\n%s: Bin-based epoching\n', Subject_ID);
    EEG = pop_epochbin(EEG, Epoch_Time, 'pre');
    EEG = eeg_checkset(EEG);
    EEG.setname = [EEG.setname '_be'];
    if (Save_All)
        EEG = pop_saveset(EEG, 'filename', [EEG.setname '.set'], 'filepath', Subject_Path);
    end
        
    % artifact detection
    fprintf('\n%s: Artifact detection\n', Subject_ID);
    EEG=pop_artmwppth(EEG, 'Channel', Moving_Window_Electrodes_1, 'Flag', [1 2], 'Review', 'off',...
        'Threshold', Moving_Window_Threshold_1, 'Twindow', Epoch_Time, 'Windowsize', Moving_Window_Width_1, 'Windowstep', Moving_Window_Step_Size_1);
    EEG=eeg_checkset(EEG);
    if N_Artifcat_Rejections > 1
        EEG=pop_artmwppth(EEG, 'Channel', Moving_Window_Electrodes_2, 'Flag', [1 3], 'Review', 'off',...
            'Threshold', Moving_Window_Threshold_2, 'Twindow', Epoch_Time, 'Windowsize', Moving_Window_Width_2, 'Windowstep', Moving_Window_Step_Size_2);
        EEG=eeg_checkset(EEG);
    end;
    EEG.setname=[EEG.setname '_ad'];
    EEG=pop_saveset(EEG, 'filename', [EEG.setname '.set'], 'filepath', Subject_Path);
    % report percent rejected across bins
    Artifact_Percent=getardetection(EEG);
    fprintf('%s: Percentage of rejected trials was %1.2f\n', Subject_ID, Artifact_Percent);
        
    % averaging (good trials)
    fprintf('\n%s: Averaging\n', Subject_ID);
    ERP=pop_averager(EEG, 'Criterion', 'good', 'DSindex', 1, 'Warning', Warnings);
    ERP.erpname=[Subject_ID '_ERPs'];
    ERP=pop_savemyerp(ERP, 'erpname', ERP.erpname, 'filename', [ERP.erpname '.erp'],...
        'filepath', Subject_Path, 'warning', Warnings);
           
    % filter ERP (low pass)
    if Highpass_Filter == 0
        fprintf('\n%s: Low-pass filtering at %g Hz\n', Subject_ID, Lowpass_Cutoff);
        ERP=pop_filterp(ERP, 1:(Channel_Number), 'Cutoff', Lowpass_Cutoff, 'Design', Lowpass_Type,...
            'Filter', 'lowpass', 'Order', Lowpass_Order);
        ERP.erpname=[ERP.erpname '_filt'];
        if (Save_All)
            ERP=pop_savemyerp(ERP, 'erpname', ERP.erpname, 'filename', [ERP.erpname '.erp'], 'filepath', Subject_Path, 'warning', Warnings);
        end
    end
        
    % create diff waves
    Diff_Waves_File = fopen(Diff_File);
    Diff_Formulas = textscan( Diff_Waves_File, '%[^\n]', 'CommentStyle','#', 'whitespace', '');
    Diff_Formulas = strtrim(cellstr(Diff_Formulas{:})');
    fclose(Diff_Waves_File);
    fprintf('\n%s: Bin operations\n', Subject_ID);
    ERP=pop_binoperator(ERP, Diff_Formulas);
    ERP.erpname=[ERP.erpname '_diff'];
    ERP=pop_savemyerp(ERP, 'erpname', ERP.erpname, 'filename', [ERP.erpname '.erp'], 'filepath', Subject_Path, 'warning', Warnings);
        
    %save final ERP in ALLERP struct
    CURRENTERP=CURRENTERP+1;
    ALLERP(CURRENTERP)=ERP;
        
    % plot erp waveforms
    pop_ploterps(ERP, Bins_to_Plot_1, Channels_to_Plot, 'LineWidth', 3, 'Style', 'Classic', 'YDir', 'reverse' );
    if Plot_Types == 2
        pop_ploterps(ERP, Bins_to_Plot_2, Channels_to_Plot, 'LineWidth', 3, 'Style', 'Classic', 'YDir', 'reverse' );
    end;
    
    % create a text file containing analysis details:
    % subject ID, analysis date, artifact rejection threshold & % rejected,
    % number of included trials per condition
    N_Trials = num2str(ERP.ntrials.accepted);
    Analysis_Date = date;
    fileID = fopen([Subject_Path Subject_ID '_details.txt'],'w');
    fprintf(fileID, 'Subject %s:\n', Subject_ID);   
    fprintf(fileID, 'Analyzed on %s\n\n', Analysis_Date);
    fprintf(fileID, 'Moving window pick-to-pick threshold: %d\n', Moving_Window_Threshold_1);
    if N_Artifcat_Rejections == 2
        fprintf(fileID, '2nd moving window pick-to-pick threshold: %d\n', Moving_Window_Threshold_2);
    end;
    fprintf(fileID, 'Percent rejected trials: %1.2f\n\n', Artifact_Percent);
    fprintf(fileID, 'Number of trials per bin:\n');
    fprintf(fileID, N_Trials, '\n');
    fclose(fileID);
    
    fprintf('\n%s: Finished!\n', Subject_ID);
        
elseif exist([Subject_Path Subject_ID '_EEG.set'], 'file') >0
    EEG = pop_loadset( 'filename', [Subject_ID '_EEG.set'], 'filepath', Subject_Path );
    EEG.setname = [Subject_ID '_EEG'];
    EEG = eeg_checkset(EEG);
    
    % create eventlist
    fprintf('\n%s: Creating eventlist\n', Subject_ID);
    EEG = pop_editeventlist(EEG, 'BoundaryNumeric', {-99}, 'BoundaryString', {'boundary'},...
        'ExportEL', [Subject_Path Subject_ID '_elist.txt'], 'List', Elist_File, 'SendEL2', 'EEG&Text', 'UpdateEEG', 'on', 'Warning', Warnings);
    EEG = eeg_checkset(EEG);
    EEG.setname = [EEG.setname '_elist'];
    if (Save_All)
        EEG = pop_saveset(EEG, 'filename', [EEG.setname '.set'], 'filepath', Subject_Path);
    end        
    
    % highpass filter (only if Highpass_Filter is set to 1)
    if Highpass_Filter
        EEG  = pop_basicfilter( EEG,  1:Channel_Number , 'Cutoff', [Highpass_Cutoff Lowpass_Cutoff],...
            'Design', Bandpass_Type, 'Filter', 'bandpass', 'Order',  Bandpass_Order );
        EEG = eeg_checkset( EEG );
        EEG.setname = [EEG.setname '_filt'];
        if (Save_All)
            EEG = pop_saveset(EEG, 'filename', [EEG.setname '.set'], 'filepath', Subject_Path);
        end
    end   
    
    % sort bins
    fprintf('\n%s: Sorting bins\n', Subject_ID);
    EEG = pop_binlister(EEG, 'BDF', Bins_File, 'ExportEL', [Subject_Path Subject_ID '_bins.txt'],...
        'ImportEL', 'no', 'Saveas', 'on', 'SendEL2', 'EEG&Text',...
        'UpdateEEG', 'on', 'Warning', Warnings);
    EEG = eeg_checkset(EEG);
    EEG.setname = [EEG.setname '_bins'];
    if (Save_All)
        EEG = pop_saveset(EEG, 'filename', [EEG.setname '.set'], 'filepath', Subject_Path);
    end
        
    % bin-based epochs
    fprintf('\n%s: Bin-based epoching\n', Subject_ID);
    EEG = pop_epochbin(EEG, Epoch_Time, 'pre');
    EEG = eeg_checkset(EEG);
    EEG.setname = [EEG.setname '_be'];
    if (Save_All)
        EEG = pop_saveset(EEG, 'filename', [EEG.setname '.set'], 'filepath', Subject_Path);
    end
        
    % artifact detection
    fprintf('\n%s: Artifact detection\n', Subject_ID);
    EEG=pop_artmwppth(EEG, 'Channel', Moving_Window_Electrodes_1, 'Flag', [1 2], 'Review', 'off',...
        'Threshold', Moving_Window_Threshold_1, 'Twindow', Epoch_Time, 'Windowsize', Moving_Window_Width_1, 'Windowstep', Moving_Window_Step_Size_1);
    EEG=eeg_checkset(EEG);
    if N_Artifcat_Rejections > 1
        EEG=pop_artmwppth(EEG, 'Channel', Moving_Window_Electrodes_2, 'Flag', [1 3], 'Review', 'off',...
            'Threshold', Moving_Window_Threshold_2, 'Twindow', Epoch_Time, 'Windowsize', Moving_Window_Width_2, 'Windowstep', Moving_Window_Step_Size_2);
        EEG=eeg_checkset(EEG);
    end;
    EEG.setname=[EEG.setname '_ad'];
    EEG=pop_saveset(EEG, 'filename', [EEG.setname '.set'], 'filepath', Subject_Path);
    % report percent rejected across bins
    Artifact_Percent=getardetection(EEG);
    fprintf('%s: Percentage of rejected trials was %1.2f\n', Subject_ID, Artifact_Percent);
        
    % averaging (good trials)
    fprintf('\n%s: Averaging\n', Subject_ID);
    ERP=pop_averager(EEG, 'Criterion', 'good', 'DSindex', 1, 'Warning', Warnings);
    ERP.erpname=[Subject_ID '_ERPs'];
    ERP=pop_savemyerp(ERP, 'erpname', ERP.erpname, 'filename', [ERP.erpname '.erp'],...
        'filepath', Subject_Path, 'warning', Warnings);
           
    % filter ERP (low pass)
    if Highpass_Filter == 0
        fprintf('\n%s: Low-pass filtering at %g Hz\n', Subject_ID, Lowpass_Cutoff);
        ERP=pop_filterp(ERP, 1:(Channel_Number), 'Cutoff', Lowpass_Cutoff, 'Design', Lowpass_Type,...
            'Filter', 'lowpass', 'Order', Lowpass_Order);
        ERP.erpname=[ERP.erpname '_filt'];
        if (Save_All)
            ERP=pop_savemyerp(ERP, 'erpname', ERP.erpname, 'filename', [ERP.erpname '.erp'], 'filepath', Subject_Path, 'warning', Warnings);
        end
    end
        
    % create diff waves
    Diff_Waves_File = fopen(Diff_File);
    Diff_Formulas = textscan( Diff_Waves_File, '%[^\n]', 'CommentStyle','#', 'whitespace', '');
    Diff_Formulas = strtrim(cellstr(Diff_Formulas{:})');
    fclose(Diff_Waves_File);
    fprintf('\n%s: Bin operations\n', Subject_ID);
    ERP=pop_binoperator(ERP, Diff_Formulas);
    ERP.erpname=[ERP.erpname '_diff'];
    ERP=pop_savemyerp(ERP, 'erpname', ERP.erpname, 'filename', [ERP.erpname '.erp'], 'filepath', Subject_Path, 'warning', Warnings);
        
    %save final ERP in ALLERP struct
    CURRENTERP=CURRENTERP+1;
    ALLERP(CURRENTERP)=ERP;
        
    % plot erp waveforms
    pop_ploterps(ERP, Bins_to_Plot_1, Channels_to_Plot, 'LineWidth', 3, 'Style', 'Classic', 'YDir', 'reverse' );
    if Plot_Types == 2
        pop_ploterps(ERP, Bins_to_Plot_2, Channels_to_Plot, 'LineWidth', 3, 'Style', 'Classic', 'YDir', 'reverse' );
    end;
    
    % create a text file containing analysis details:
    % subject ID, analysis date, artifact rejection threshold & % rejected,
    % number of included trials per condition
    N_Trials = num2str(ERP.ntrials.accepted);
    Analysis_Date = date;
    fileID = fopen([Subject_Path Subject_ID '_details.txt'],'w');
    fprintf(fileID, 'Subject %s:\n', Subject_ID);   
    fprintf(fileID, 'Analyzed on %s\n\n', Analysis_Date);
    fprintf(fileID, 'Moving window pick-to-pick threshold: %d\n', Moving_Window_Threshold_1);
    if N_Artifcat_Rejections == 2
        fprintf(fileID, '2nd moving window pick-to-pick threshold: %d\n', Moving_Window_Threshold_2);
    end;
    fprintf(fileID, 'Percent rejected trials: %1.2f\n\n', Artifact_Percent);
    fprintf(fileID, 'Number of trials per bin:\n');
    fprintf(fileID, N_Trials, '\n');
    fclose(fileID);
    
    fprintf('\n%s: Finished!\n', Subject_ID);
      
else
    fprintf('\nWhoops! %s doesn''t exist!', Subject_ID);       
end;